<?php
/**
 * Plugin Name: Sumo Subscriptions Cleanup
 * Plugin URI: https://foxco.net
 * Description: Removes leftover database entries from Sumo Subscriptions plugin
 * Version: 1.0.0
 * Author: Foxco.net
 * Author URI: https://foxco.net
 * License: GPL v2 or later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 */

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

class Sumo_Cleanup {
    
    private $options_to_remove = array();
    private $tables_to_remove = array();
    private $meta_keys_to_remove = array();
    
    public function __construct() {
        add_action('admin_menu', array($this, 'add_admin_menu'));
        add_action('admin_post_sumo_cleanup_scan', array($this, 'scan_database'));
        add_action('admin_post_sumo_cleanup_remove', array($this, 'remove_data'));
        
        // Common Sumo-related prefixes and patterns
        $this->sumo_patterns = array(
            'sumo',
            'sumome',
            'sumo_',
            'sumome_',
            '_sumo',
            'sumome-'
        );
    }
    
    public function add_admin_menu() {
        add_management_page(
            'Sumo Cleanup',
            'Sumo Cleanup',
            'manage_options',
            'sumo-cleanup',
            array($this, 'admin_page')
        );
    }
    
    public function admin_page() {
        ?>
        <div class="wrap">
            <h1>🦊 Sumo Subscriptions Database Cleanup</h1>
            <p style="font-size: 14px; color: #666;">by <a href="https://foxco.net" target="_blank">Foxco.net</a></p>
            
            <?php if (isset($_GET['scan']) && $_GET['scan'] === 'complete'): ?>
                <div class="notice notice-success">
                    <p>Database scan complete! Review the findings below.</p>
                </div>
            <?php endif; ?>
            
            <?php if (isset($_GET['removed']) && $_GET['removed'] === 'success'): ?>
                <div class="notice notice-success">
                    <p>Sumo data has been successfully removed from your database!</p>
                </div>
            <?php endif; ?>
            
            <div class="card">
                <h2>Step 1: Scan Database</h2>
                <p>Click below to scan your database for leftover Sumo Subscriptions data.</p>
                <form method="post" action="<?php echo admin_url('admin-post.php'); ?>">
                    <input type="hidden" name="action" value="sumo_cleanup_scan">
                    <?php wp_nonce_field('sumo_cleanup_scan'); ?>
                    <button type="submit" class="button button-primary">Scan Database</button>
                </form>
            </div>
            
            <?php
            $scan_results = get_transient('sumo_cleanup_scan_results');
            if ($scan_results):
            ?>
            <div class="card" style="margin-top: 20px;">
                <h2>Step 2: Review Findings</h2>
                
                <h3>Options Found: <?php echo count($scan_results['options']); ?></h3>
                <?php if (!empty($scan_results['options'])): ?>
                    <ul style="max-height: 200px; overflow-y: auto; background: #f5f5f5; padding: 10px; border: 1px solid #ddd;">
                        <?php foreach ($scan_results['options'] as $option): ?>
                            <li><code><?php echo esc_html($option); ?></code></li>
                        <?php endforeach; ?>
                    </ul>
                <?php else: ?>
                    <p>No Sumo options found.</p>
                <?php endif; ?>
                
                <h3>Custom Tables Found: <?php echo count($scan_results['tables']); ?></h3>
                <?php if (!empty($scan_results['tables'])): ?>
                    <ul style="background: #f5f5f5; padding: 10px; border: 1px solid #ddd;">
                        <?php foreach ($scan_results['tables'] as $table): ?>
                            <li><code><?php echo esc_html($table); ?></code></li>
                        <?php endforeach; ?>
                    </ul>
                <?php else: ?>
                    <p>No Sumo tables found.</p>
                <?php endif; ?>
                
                <h3>Post Meta Keys Found: <?php echo count($scan_results['post_meta']); ?></h3>
                <?php if (!empty($scan_results['post_meta'])): ?>
                    <ul style="max-height: 200px; overflow-y: auto; background: #f5f5f5; padding: 10px; border: 1px solid #ddd;">
                        <?php foreach ($scan_results['post_meta'] as $meta): ?>
                            <li><code><?php echo esc_html($meta); ?></code></li>
                        <?php endforeach; ?>
                    </ul>
                <?php else: ?>
                    <p>No Sumo post meta found.</p>
                <?php endif; ?>
                
                <h3>User Meta Keys Found: <?php echo count($scan_results['user_meta']); ?></h3>
                <?php if (!empty($scan_results['user_meta'])): ?>
                    <ul style="max-height: 200px; overflow-y: auto; background: #f5f5f5; padding: 10px; border: 1px solid #ddd;">
                        <?php foreach ($scan_results['user_meta'] as $meta): ?>
                            <li><code><?php echo esc_html($meta); ?></code></li>
                        <?php endforeach; ?>
                    </ul>
                <?php else: ?>
                    <p>No Sumo user meta found.</p>
                <?php endif; ?>
                
                <h3>Total Items: <?php 
                    echo count($scan_results['options']) + 
                         count($scan_results['tables']) + 
                         count($scan_results['post_meta']) + 
                         count($scan_results['user_meta']); 
                ?></h3>
                
                <?php if (count($scan_results['options']) + count($scan_results['tables']) + count($scan_results['post_meta']) + count($scan_results['user_meta']) > 0): ?>
                    <div style="margin-top: 20px; padding: 15px; background: #fff3cd; border: 1px solid #ffc107;">
                        <h3 style="margin-top: 0; color: #856404;">⚠️ Warning</h3>
                        <p><strong>This action cannot be undone!</strong> Please make sure you have a database backup before proceeding.</p>
                        <p>It's recommended to backup your database using your hosting control panel or a plugin like UpdraftPlus.</p>
                    </div>
                    
                    <form method="post" action="<?php echo admin_url('admin-post.php'); ?>" style="margin-top: 20px;">
                        <input type="hidden" name="action" value="sumo_cleanup_remove">
                        <?php wp_nonce_field('sumo_cleanup_remove'); ?>
                        <label>
                            <input type="checkbox" name="confirm" value="1" required>
                            I have backed up my database and want to proceed
                        </label>
                        <br><br>
                        <button type="submit" class="button button-primary button-large" style="background: #dc3545; border-color: #dc3545;">
                            Remove All Sumo Data
                        </button>
                    </form>
                <?php endif; ?>
            </div>
            <?php endif; ?>
            
            <div style="margin-top: 30px; padding: 20px; text-align: center; background: #f9f9f9; border: 1px solid #ddd;">
                <p style="margin: 0; color: #666;">
                    Made with care by <a href="https://foxco.net" target="_blank" style="color: #ff6b35; text-decoration: none; font-weight: bold;">🦊 Foxco.net</a>
                </p>
            </div>
        </div>
        <style>
            .card { background: white; padding: 20px; border: 1px solid #ccd0d4; box-shadow: 0 1px 1px rgba(0,0,0,.04); }
            .card h2 { margin-top: 0; }
        </style>
        <?php
    }
    
    public function scan_database() {
        check_admin_referer('sumo_cleanup_scan');
        
        if (!current_user_can('manage_options')) {
            wp_die('Unauthorized');
        }
        
        global $wpdb;
        
        $results = array(
            'options' => array(),
            'tables' => array(),
            'post_meta' => array(),
            'user_meta' => array()
        );
        
        // Scan options
        foreach ($this->sumo_patterns as $pattern) {
            $options = $wpdb->get_col($wpdb->prepare(
                "SELECT option_name FROM {$wpdb->options} WHERE option_name LIKE %s",
                '%' . $wpdb->esc_like($pattern) . '%'
            ));
            $results['options'] = array_merge($results['options'], $options);
        }
        $results['options'] = array_unique($results['options']);
        
        // Scan for custom tables
        $all_tables = $wpdb->get_col("SHOW TABLES LIKE '{$wpdb->prefix}%'");
        foreach ($all_tables as $table) {
            foreach ($this->sumo_patterns as $pattern) {
                if (stripos($table, $pattern) !== false) {
                    $results['tables'][] = $table;
                    break;
                }
            }
        }
        
        // Scan post meta
        foreach ($this->sumo_patterns as $pattern) {
            $post_meta = $wpdb->get_col($wpdb->prepare(
                "SELECT DISTINCT meta_key FROM {$wpdb->postmeta} WHERE meta_key LIKE %s",
                '%' . $wpdb->esc_like($pattern) . '%'
            ));
            $results['post_meta'] = array_merge($results['post_meta'], $post_meta);
        }
        $results['post_meta'] = array_unique($results['post_meta']);
        
        // Scan user meta
        foreach ($this->sumo_patterns as $pattern) {
            $user_meta = $wpdb->get_col($wpdb->prepare(
                "SELECT DISTINCT meta_key FROM {$wpdb->usermeta} WHERE meta_key LIKE %s",
                '%' . $wpdb->esc_like($pattern) . '%'
            ));
            $results['user_meta'] = array_merge($results['user_meta'], $user_meta);
        }
        $results['user_meta'] = array_unique($results['user_meta']);
        
        set_transient('sumo_cleanup_scan_results', $results, HOUR_IN_SECONDS);
        
        wp_redirect(add_query_arg('scan', 'complete', admin_url('tools.php?page=sumo-cleanup')));
        exit;
    }
    
    public function remove_data() {
        check_admin_referer('sumo_cleanup_remove');
        
        if (!current_user_can('manage_options')) {
            wp_die('Unauthorized');
        }
        
        if (!isset($_POST['confirm']) || $_POST['confirm'] != '1') {
            wp_die('You must confirm the action');
        }
        
        global $wpdb;
        
        $scan_results = get_transient('sumo_cleanup_scan_results');
        
        if (!$scan_results) {
            wp_die('No scan results found. Please scan the database first.');
        }
        
        // Remove options
        foreach ($scan_results['options'] as $option) {
            delete_option($option);
        }
        
        // Drop custom tables
        foreach ($scan_results['tables'] as $table) {
            $wpdb->query("DROP TABLE IF EXISTS `{$table}`");
        }
        
        // Remove post meta
        foreach ($scan_results['post_meta'] as $meta_key) {
            delete_post_meta_by_key($meta_key);
        }
        
        // Remove user meta
        foreach ($scan_results['user_meta'] as $meta_key) {
            $wpdb->delete($wpdb->usermeta, array('meta_key' => $meta_key));
        }
        
        // Clear the scan results
        delete_transient('sumo_cleanup_scan_results');
        
        wp_redirect(add_query_arg('removed', 'success', admin_url('tools.php?page=sumo-cleanup')));
        exit;
    }
}

// Initialize the plugin
new Sumo_Cleanup();
