<?php
/**
 * Plugin Name: 🦊 Foxco Divi Content Cleaner
 * Description: Professional Divi Page Builder content cleaning tool by Foxco.net - Remove all Divi shortcodes while preserving your content
 * Version: 1.0.0
 * Author: Foxco.net
 * Author URI: https://foxco.net
 * License: GPL v2 or later
 */

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

class FoxcoDiviContentCleaner {
    
    public function __construct() {
        add_action('admin_menu', array($this, 'add_admin_menu'));
        add_action('wp_ajax_foxco_divi_cleaner_preview', array($this, 'ajax_preview'));
        add_action('wp_ajax_foxco_divi_cleaner_process', array($this, 'ajax_process'));
        add_action('admin_enqueue_scripts', array($this, 'enqueue_scripts'));
    }
    
    public function add_admin_menu() {
        add_menu_page(
            '🦊 Foxco Divi Cleaner',
            '🦊 Divi Cleaner',
            'manage_options',
            'foxco-divi-cleaner',
            array($this, 'admin_page'),
            'data:image/svg+xml;base64,' . base64_encode('<svg width="20" height="20" viewBox="0 0 20 20" xmlns="http://www.w3.org/2000/svg"><text y="15" font-size="16">🦊</text></svg>'),
            30
        );
    }
    
    public function enqueue_scripts($hook) {
        if ($hook !== 'toplevel_page_foxco-divi-cleaner') {
            return;
        }
        
        wp_enqueue_script('jquery');
        wp_localize_script('jquery', 'foxco_divi_cleaner_ajax', array(
            'ajax_url' => admin_url('admin-ajax.php'),
            'nonce' => wp_create_nonce('foxco_divi_cleaner_nonce')
        ));
    }
    
    public function admin_page() {
        ?>
        <div class="wrap foxco-divi-cleaner">
            <div class="foxco-header">
                <h1>🦊 Foxco Divi Content Cleaner</h1>
                <p class="foxco-subtitle">Professional Divi content cleaning by <a href="https://foxco.net" target="_blank">Foxco.net</a></p>
            </div>
            
            <div class="notice notice-warning foxco-warning">
                <p><strong>🦊 ⚠️ Important:</strong> Always backup your database before using this tool! This process is irreversible. Foxco.net recommends testing on a staging site first.</p>
            </div>
            
            <div class="foxco-card">
                <h2>🦊 What Foxco's Divi Cleaner Does:</h2>
                <ul class="foxco-features">
                    <li>🧹 Removes all Divi Page Builder shortcodes (et_pb_section, et_pb_row, et_pb_column, etc.)</li>
                    <li>💾 Preserves the actual text content inside these shortcodes</li>
                    <li>🔄 Converts some Divi elements to standard HTML (headings, paragraphs, etc.)</li>
                    <li>✨ Cleans up empty paragraphs and extra whitespace</li>
                    <li>🎯 Professional-grade content migration tool</li>
                </ul>
            </div>
            
            <div class="foxco-card">
                <h2>🦊 Step 1: Preview Changes</h2>
                <p>Select post types to clean with Foxco's advanced algorithm:</p>
                <div class="foxco-checkbox-group">
                    <label class="foxco-checkbox"><input type="checkbox" id="clean_posts" checked> <span>📝 Posts</span></label>
                    <label class="foxco-checkbox"><input type="checkbox" id="clean_pages"> <span>📄 Pages</span></label>
                </div>
                
                <button id="preview-btn" class="button button-primary foxco-btn-primary">🦊 Preview Changes</button>
                <div id="preview-results" class="foxco-results"></div>
            </div>
            
            <div class="foxco-card" id="process-section" style="display: none;">
                <h2>🦊 Step 2: Process Posts</h2>
                <div class="foxco-warning-box">
                    <p><strong>🦊 ⚠️ This will permanently modify your posts!</strong></p>
                    <p>Foxco.net strongly recommends having a complete database backup before proceeding.</p>
                </div>
                <button id="process-btn" class="button button-secondary foxco-btn-secondary">🦊 Clean All Selected Posts</button>
                <div id="process-results" class="foxco-results"></div>
            </div>
            
            <div class="foxco-footer">
                <p>🦊 <strong>Foxco Divi Content Cleaner</strong> - Professional WordPress development tools by <a href="https://foxco.net" target="_blank">Foxco.net</a></p>
                <p>Need custom WordPress development? <a href="https://foxco.net" target="_blank">Contact Foxco.net</a> for professional services.</p>
            </div>
        </div>
        
        <script>
        jQuery(document).ready(function($) {
            $('#preview-btn').click(function() {
                var button = $(this);
                var originalText = button.html();
                button.html('🦊 Loading...').prop('disabled', true);
                
                var post_types = [];
                if ($('#clean_posts').is(':checked')) post_types.push('post');
                if ($('#clean_pages').is(':checked')) post_types.push('page');
                
                $.ajax({
                    url: foxco_divi_cleaner_ajax.ajax_url,
                    type: 'POST',
                    data: {
                        action: 'foxco_divi_cleaner_preview',
                        nonce: foxco_divi_cleaner_ajax.nonce,
                        post_types: post_types
                    },
                    success: function(response) {
                        $('#preview-results').html(response);
                        $('#process-section').show();
                        button.html(originalText).prop('disabled', false);
                    },
                    error: function() {
                        alert('🦊 Foxco Error: Something went wrong during preview. Please try again.');
                        button.html(originalText).prop('disabled', false);
                    }
                });
            });
            
            $('#process-btn').click(function() {
                if (!confirm('🦊 Foxco Warning: Are you sure? This will permanently modify your posts. Make sure you have a backup! Foxco.net is not responsible for data loss.')) {
                    return;
                }
                
                var button = $(this);
                var originalText = button.html();
                button.html('🦊 Processing...').prop('disabled', true);
                
                var post_types = [];
                if ($('#clean_posts').is(':checked')) post_types.push('post');
                if ($('#clean_pages').is(':checked')) post_types.push('page');
                
                $.ajax({
                    url: foxco_divi_cleaner_ajax.ajax_url,
                    type: 'POST',
                    data: {
                        action: 'foxco_divi_cleaner_process',
                        nonce: foxco_divi_cleaner_ajax.nonce,
                        post_types: post_types
                    },
                    success: function(response) {
                        $('#process-results').html(response);
                        button.html('🦊 Completed').prop('disabled', true);
                    },
                    error: function() {
                        alert('🦊 Foxco Error: Something went wrong during processing. Please check your posts manually.');
                        button.html(originalText).prop('disabled', false);
                    }
                });
            });
        });
        </script>
        
        <style>
        .foxco-divi-cleaner {
            font-family: -apple-system, BlinkMacSystemFont, "Segoe UI", Roboto, sans-serif;
        }
        
        .foxco-header {
            background: linear-gradient(135deg, #ff6b35, #f7931e);
            color: white;
            padding: 30px;
            border-radius: 8px;
            margin-bottom: 20px;
            text-align: center;
        }
        
        .foxco-header h1 {
            color: white;
            font-size: 2.5em;
            margin: 0;
            text-shadow: 2px 2px 4px rgba(0,0,0,0.3);
        }
        
        .foxco-subtitle {
            font-size: 1.1em;
            margin: 10px 0 0 0;
            opacity: 0.9;
        }
        
        .foxco-subtitle a {
            color: white;
            text-decoration: underline;
            font-weight: bold;
        }
        
        .foxco-card {
            background: white;
            border: 2px solid #ff6b35;
            border-radius: 8px;
            padding: 25px;
            margin: 20px 0;
            box-shadow: 0 4px 6px rgba(0, 0, 0, 0.1);
        }
        
        .foxco-card h2 {
            color: #ff6b35;
            border-bottom: 2px solid #ff6b35;
            padding-bottom: 10px;
            margin-top: 0;
        }
        
        .foxco-features {
            list-style: none;
            padding: 0;
        }
        
        .foxco-features li {
            padding: 8px 0;
            font-size: 1.1em;
        }
        
        .foxco-checkbox-group {
            margin: 20px 0;
        }
        
        .foxco-checkbox {
            display: block;
            margin: 10px 0;
            font-size: 1.1em;
            cursor: pointer;
        }
        
        .foxco-checkbox input {
            margin-right: 10px;
        }
        
        .foxco-btn-primary {
            background: linear-gradient(135deg, #ff6b35, #f7931e) !important;
            border: none !important;
            color: white !important;
            padding: 12px 24px !important;
            font-size: 1.1em !important;
            border-radius: 6px !important;
            font-weight: bold !important;
            text-shadow: 1px 1px 2px rgba(0,0,0,0.2) !important;
        }
        
        .foxco-btn-primary:hover {
            background: linear-gradient(135deg, #e55a2b, #e8841a) !important;
        }
        
        .foxco-btn-secondary {
            background: #666 !important;
            border: none !important;
            color: white !important;
            padding: 12px 24px !important;
            font-size: 1.1em !important;
            border-radius: 6px !important;
            font-weight: bold !important;
        }
        
        .foxco-btn-secondary:hover {
            background: #555 !important;
        }
        
        .foxco-warning {
            border-left-color: #ff6b35 !important;
        }
        
        .foxco-warning-box {
            background: #fff3cd;
            border: 1px solid #ffeaa7;
            border-radius: 6px;
            padding: 15px;
            margin: 15px 0;
        }
        
        .foxco-results {
            margin-top: 20px;
        }
        
        .foxco-preview-item {
            border: 2px solid #ff6b35;
            margin: 15px 0;
            padding: 20px;
            background: #fff8f5;
            border-radius: 6px;
        }
        
        .foxco-preview-item h4 {
            color: #ff6b35;
            margin-top: 0;
        }
        
        .foxco-before-after {
            display: grid;
            grid-template-columns: 1fr 1fr;
            gap: 20px;
            margin-top: 15px;
        }
        
        .foxco-before, .foxco-after {
            padding: 15px;
            background: white;
            border: 1px solid #ddd;
            border-radius: 4px;
            max-height: 200px;
            overflow-y: auto;
            font-family: 'Monaco', 'Courier New', monospace;
            font-size: 12px;
        }
        
        .foxco-before h4, .foxco-after h4 {
            margin: 0 0 10px 0;
            font-family: -apple-system, BlinkMacSystemFont, "Segoe UI", Roboto, sans-serif;
            color: #333;
        }
        
        .foxco-footer {
            background: #f8f9fa;
            border: 1px solid #dee2e6;
            border-radius: 8px;
            padding: 20px;
            text-align: center;
            margin-top: 30px;
            color: #666;
        }
        
        .foxco-footer a {
            color: #ff6b35;
            text-decoration: none;
            font-weight: bold;
        }
        
        .foxco-footer a:hover {
            text-decoration: underline;
        }
        
        .foxco-success {
            background: #d4edda;
            border: 1px solid #c3e6cb;
            color: #155724;
            padding: 15px;
            border-radius: 6px;
            margin: 15px 0;
        }
        </style>
        <?php
    }
    
    public function ajax_preview() {
        check_ajax_referer('foxco_divi_cleaner_nonce', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_die('🦊 Foxco Error: Insufficient permissions');
        }
        
        $post_types = isset($_POST['post_types']) ? $_POST['post_types'] : array('post');
        
        $posts = get_posts(array(
            'post_type' => $post_types,
            'posts_per_page' => 10, // Limit for preview
            'post_status' => 'any',
            'meta_query' => array(
                array(
                    'key' => '_et_pb_use_builder',
                    'value' => 'on',
                    'compare' => '='
                )
            )
        ));
        
        if (empty($posts)) {
            echo '<div class="notice notice-info"><p>🦊 <strong>Foxco Report:</strong> No posts found with Divi content. Your site is already clean!</p></div>';
            wp_die();
        }
        
        echo '<h3>🦊 Foxco Preview - Showing first 10 posts with Divi content:</h3>';
        
        foreach ($posts as $post) {
            $original_content = $post->post_content;
            $cleaned_content = $this->clean_divi_content($original_content);
            
            if ($original_content !== $cleaned_content) {
                echo '<div class="foxco-preview-item">';
                echo '<h4>🦊 ' . esc_html($post->post_title) . ' (ID: ' . $post->ID . ')</h4>';
                echo '<div class="foxco-before-after">';
                echo '<div class="foxco-before"><h4>Before Foxco Cleaning:</h4>' . esc_html(substr($original_content, 0, 500)) . '...</div>';
                echo '<div class="foxco-after"><h4>After Foxco Cleaning:</h4>' . esc_html(substr($cleaned_content, 0, 500)) . '...</div>';
                echo '</div>';
                echo '</div>';
            }
        }
        
        $total_count = $this->get_divi_posts_count($post_types);
        echo '<div class="foxco-success">';
        echo '<p><strong>🦊 Foxco Analysis Complete!</strong></p>';
        echo '<p>Total posts to be processed by Foxco: <strong>' . $total_count . '</strong></p>';
        echo '</div>';
        
        wp_die();
    }
    
    public function ajax_process() {
        check_ajax_referer('foxco_divi_cleaner_nonce', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_die('🦊 Foxco Error: Insufficient permissions');
        }
        
        $post_types = isset($_POST['post_types']) ? $_POST['post_types'] : array('post');
        
        $posts = get_posts(array(
            'post_type' => $post_types,
            'posts_per_page' => -1,
            'post_status' => 'any',
            'meta_query' => array(
                array(
                    'key' => '_et_pb_use_builder',
                    'value' => 'on',
                    'compare' => '='
                )
            )
        ));
        
        $processed = 0;
        $errors = 0;
        
        foreach ($posts as $post) {
            $original_content = $post->post_content;
            $cleaned_content = $this->clean_divi_content($original_content);
            
            if ($original_content !== $cleaned_content) {
                $result = wp_update_post(array(
                    'ID' => $post->ID,
                    'post_content' => $cleaned_content
                ));
                
                if ($result) {
                    // Remove Divi meta fields
                    delete_post_meta($post->ID, '_et_pb_use_builder');
                    delete_post_meta($post->ID, '_et_pb_old_content');
                    $processed++;
                } else {
                    $errors++;
                }
            }
        }
        
        echo '<div class="foxco-success">';
        echo '<h3>🦊 Foxco Processing Complete!</h3>';
        echo '<p><strong>Posts successfully cleaned by Foxco:</strong> ' . $processed . '</p>';
        if ($errors > 0) {
            echo '<p><strong>Errors encountered:</strong> ' . $errors . '</p>';
            echo '<p>🦊 <em>Contact Foxco.net if you need assistance with the errors.</em></p>';
        }
        echo '<p>🦊 <strong>Your content has been professionally cleaned by Foxco.net!</strong></p>';
        echo '</div>';
        
        wp_die();
    }
    
    private function get_divi_posts_count($post_types) {
        $posts = get_posts(array(
            'post_type' => $post_types,
            'posts_per_page' => -1,
            'post_status' => 'any',
            'fields' => 'ids',
            'meta_query' => array(
                array(
                    'key' => '_et_pb_use_builder',
                    'value' => 'on',
                    'compare' => '='
                )
            )
        ));
        
        return count($posts);
    }
    
    private function clean_divi_content($content) {
        // Foxco's advanced Divi content cleaning algorithm
        // Remove all Divi shortcodes but preserve content inside them
        $patterns = array(
            // Text modules - preserve content
            '/\[et_pb_text[^\]]*\](.*?)\[\/et_pb_text\]/s' => '$1',
            
            // Heading modules - convert to HTML headings
            '/\[et_pb_text[^\]]*admin_label="[^"]*[Hh]eading[^"]*"[^\]]*\]<h([1-6])[^>]*>(.*?)<\/h[1-6]>\[\/et_pb_text\]/s' => '<h$1>$2</h$1>',
            
            // Image modules - preserve img tags
            '/\[et_pb_image[^\]]*\](.*?)\[\/et_pb_image\]/s' => '$1',
            
            // Button modules - convert to simple links
            '/\[et_pb_button[^\]]*button_url="([^"]*)"[^\]]*button_text="([^"]*)"[^\]]*\]\[\/et_pb_button\]/s' => '<a href="$1">$2</a>',
            
            // Blurb modules - preserve content
            '/\[et_pb_blurb[^\]]*\](.*?)\[\/et_pb_blurb\]/s' => '$1',
            
            // Accordion modules - preserve content
            '/\[et_pb_accordion[^\]]*\](.*?)\[\/et_pb_accordion\]/s' => '$1',
            '/\[et_pb_accordion_item[^\]]*title="([^"]*)"[^\]]*\](.*?)\[\/et_pb_accordion_item\]/s' => '<h4>$1</h4>$2',
            
            // Tabs modules - preserve content
            '/\[et_pb_tabs[^\]]*\](.*?)\[\/et_pb_tabs\]/s' => '$1',
            '/\[et_pb_tab[^\]]*title="([^"]*)"[^\]]*\](.*?)\[\/et_pb_tab\]/s' => '<h4>$1</h4>$2',
            
            // Contact form - preserve content but simplify
            '/\[et_pb_contact_form[^\]]*\](.*?)\[\/et_pb_contact_form\]/s' => '<div class="contact-form">$1</div>',
            
            // Video modules - preserve content
            '/\[et_pb_video[^\]]*\](.*?)\[\/et_pb_video\]/s' => '$1',
            
            // Code modules - preserve content
            '/\[et_pb_code[^\]]*\](.*?)\[\/et_pb_code\]/s' => '$1',
            
            // Columns, rows, sections - just remove wrappers
            '/\[et_pb_section[^\]]*\]/s' => '',
            '/\[\/et_pb_section\]/s' => '',
            '/\[et_pb_row[^\]]*\]/s' => '',
            '/\[\/et_pb_row\]/s' => '',
            '/\[et_pb_column[^\]]*\]/s' => '',
            '/\[\/et_pb_column\]/s' => '',
            
            // Remove any remaining Divi shortcodes
            '/\[et_pb_[^\]]*\]/s' => '',
            '/\[\/et_pb_[^\]]*\]/s' => '',
        );
        
        // Apply Foxco's multi-pass cleaning algorithm for nested shortcodes
        for ($i = 0; $i < 5; $i++) {
            $old_content = $content;
            foreach ($patterns as $pattern => $replacement) {
                $content = preg_replace($pattern, $replacement, $content);
            }
            // If no changes were made, break the loop
            if ($old_content === $content) {
                break;
            }
        }
        
        // Foxco's advanced content cleanup
        $content = preg_replace('/\s+/', ' ', $content); // Multiple spaces to single
        $content = preg_replace('/<p>\s*<\/p>/', '', $content); // Empty paragraphs
        $content = preg_replace('/(<br\s*\/?>){3,}/', '<br><br>', $content); // Multiple line breaks
        $content = trim($content);
        
        // Ensure paragraphs for content
        if (!empty($content) && !preg_match('/^<[^>]+>/', $content)) {
            $content = '<p>' . $content . '</p>';
        }
        
        return $content;
    }
}

// Initialize Foxco's Divi Content Cleaner
new FoxcoDiviContentCleaner();
?>